﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using Microsoft.Kinect;
using Microsoft.Kinect.Toolkit;
using Microsoft.Speech.AudioFormat;
using Microsoft.Speech.Recognition;

namespace KinectCameraSample
{
    /// <summary>
    /// MainWindow.xaml の相互作用ロジック
    /// </summary>
    public partial class MainWindow : Window
    {
        // RGBカメラの解像度・フレームレート
        //ColorImageFormat rgbFormat = ColorImageFormat.RgbResolution640x480Fps30;

        // KinectSensorChooser
        private KinectSensorChooser kinectChooser = new KinectSensorChooser();

        // ジェスチャー認識の際の遊び
        private const double MOVE_PLAY = 0.1;

        // ジェスチャーの途中経過の状態
        private int[] gestureState;

        // 直前の姿勢を取得するための骨格情報
        private Skeleton[] prevSkeleton;

        // カメラ・深度センサーの解像度・フレームレート
        private const ColorImageFormat rgbFormat
            = ColorImageFormat.RgbResolution640x480Fps30;
        private const DepthImageFormat depthFormat
            = DepthImageFormat.Resolution640x480Fps30;
        //  = DepthImageFormat.Resolution320x240Fps30;

        // Kinectセンサーからの画像情報を受け取るバッファ
        private byte[] pixelBuffer = null;

        // kinectセンサーからの深度情報を受け取るバッファ
        private short[] depthBuffer = null;

        // 深度情報の各点に対する画像情報上の座標
        private ColorImagePoint[] clrPntBuffer = null;

        // 深度情報で背景を覆う画像のデータ
        private byte[] depMaskBuffer = null;

        // Kinectセンサーからの骨格情報を受け取るバッファ
        private Skeleton[] skeletonBuffer = null;

        // 画面に表示するビットマップ
        private RenderTargetBitmap bmpBuffer = null;

        // 顔のビットマップイメージ
        private BitmapImage maskImage = null;
        private BitmapImage fukidashiImage = null;

        // 音源の角度
        private double soundDir = double.NaN;

        // ビットマップへの描画用DrawingVisual
        private DrawingVisual drawVisual = new DrawingVisual();

        // 音声認識エンジン
        private SpeechRecognitionEngine speechEngine;

        // 認識された文
        private string recognizedText = null;

        public MainWindow()
        {
            InitializeComponent();
        }

        private void WindowLoaded(object sender, RoutedEventArgs e)
        {
            // Kinectセンサーの取得(エラー処理など省略版)
            KinectSensor kinect = KinectSensor.KinectSensors[0];

            // 画像の読み込み
            Uri imgUri = new Uri("pack://application:,,,/images/goya.png");
            maskImage = new BitmapImage(imgUri);
            Uri fkuri = new Uri("pack://application:,,,/images/fukidashi.gif");
            fukidashiImage = new BitmapImage(fkuri);

            kinectChooser.KinectChanged += KinectChanged;
            kinectChooser.PropertyChanged += KinectPropChanged;
            kinectChooser.Start();
        }

        //Kinectセンターの初期化
        private void InitKinectSensor(KinectSensor kinect)
        {
            // ストリームの有効化
            ColorImageStream clrStream = kinect.ColorStream;
            clrStream.Enable(rgbFormat);
            DepthImageStream depStream = kinect.DepthStream;
            depStream.Enable(depthFormat);
            SkeletonStream skelStream = kinect.SkeletonStream;
            skelStream.Enable();

            // バッファの初期化
            pixelBuffer = new byte[clrStream.FramePixelDataLength];
            depthBuffer = new short[depStream.FramePixelDataLength];
            clrPntBuffer = new ColorImagePoint[depStream.FramePixelDataLength];
            depMaskBuffer = new byte[clrStream.FramePixelDataLength];
            skeletonBuffer = new Skeleton[skelStream.FrameSkeletonArrayLength];
            gestureState = new int[skelStream.FrameSkeletonArrayLength]; //10回ジェスチャー
            prevSkeleton = new Skeleton[skelStream.FrameSkeletonArrayLength]; //10回ジェスチャー
            bmpBuffer = new RenderTargetBitmap(clrStream.FrameWidth,
                                               clrStream.FrameHeight,
                                               98, 96, PixelFormats.Default);

            rgbImage.Source = bmpBuffer;

            // 音声認識エンジンの初期化
            speechEngine = InitSpeechEngine();

            // イベントハンドラの登録
            kinect.AllFramesReady += AllFramesReady;
            kinect.AudioSource.SoundSourceAngleChanged += SoundSrcAngleChanged;
            speechEngine.SpeechRecognized += SpeechRecognized;

            // Kinectセンサーからのストリーム取得を開始
            //kinect.Start();
            //kinect.AudioSource.Start();
            System.IO.Stream stream = kinect.AudioSource.Start();
            var speechAudioFormat
                = new SpeechAudioFormatInfo(EncodingFormat.Pcm, 16000, 16, 1,
                                            32000, 2, null);
            speechEngine.SetInputToAudioStream(stream, speechAudioFormat);
            speechEngine.RecognizeAsync(RecognizeMode.Multiple);

        }

        // Kinectセンサーの終了処理
        private void UninitKinectSensor(KinectSensor kinect)
        {
            kinect.AllFramesReady -= AllFramesReady;
            kinect.AudioSource.SoundSourceAngleChanged -= SoundSrcAngleChanged;
            UninitSpeechEngine();
        }

        // Kinectセンサーの挿抜イベントに対し、初期化/終了処理を呼び出す
        private void KinectChanged(object sender, KinectChangedEventArgs args)
        {
            if (args.OldSensor != null)
                UninitKinectSensor(args.OldSensor);

            if (args.NewSensor != null)
                InitKinectSensor(args.NewSensor);
        }

        private void KinectPropChanged(Object sendor,
                                       PropertyChangedEventArgs args)
        {
            if ("Status".Equals(args.PropertyName))
            {
                textBlockStatus.Text = "Status: " + kinectChooser.Status;
            }
        }

        // SpeechRecognizedイベントのハンドラ
        private void SpeechRecognized(object sender,
                                      SpeechRecognizedEventArgs e)
        {
            if (e.Result != null && e.Result.Confidence >= 0.3)
                recognizedText = e.Result.Text;
            else
                recognizedText = null;
        }

        // SoundSourceAngleChanged イベントのハンドラ
        // (確度が一定以上の場合だけ確度を記録)
        private void SoundSrcAngleChanged(object sender,
                                          SoundSourceAngleChangedEventArgs e)
        {
            soundDir = e.ConfidenceLevel > 0.5 ? e.Angle : double.NaN;
        }

        // FrameReady イベントのハンドラ
        // (画像情報を取得・顔の部分にマスクを上書きして描画)
        private void AllFramesReady(object sender, AllFramesReadyEventArgs e)
        {
            KinectSensor kinect = sender as KinectSensor;
            //List<SkeletonPoint> headList = null;
            List<Tuple<SkeletonPoint, Matrix4, int>> headList = null;

            // 骨格情報から、頭の座標リストを作成
            using (SkeletonFrame skeletonFrame = e.OpenSkeletonFrame())
            {
                if (skeletonFrame != null)
                    headList = getHeadPoints(skeletonFrame);
            }

            // カメラの画像情報に、顔の位置にマスクを上書きして描画
            using (ColorImageFrame imageFrame = e.OpenColorImageFrame())
            using (DepthImageFrame depthFrame = e.OpenDepthImageFrame())
            {
                if (imageFrame != null)
                    //fillBitmap(kinect, imageFrame, headList);
                    fillBitmap(kinect, imageFrame, depthFrame, headList);
            }
        }

        // 骨格情報から、頭の位置を取得しリストに入れて返す
        private List<Tuple<SkeletonPoint, Matrix4, int>> getHeadPoints(SkeletonFrame skelFrame)
        {
            // 処理結果のリストを空の状態で作成
            //List<SkeletonPoint> results = new List<SkeletonPoint>();
            var results = new List<Tuple<SkeletonPoint, Matrix4, int>>();

            // 骨格情報をバッファにコピー
            skelFrame.CopySkeletonDataTo(skeletonBuffer);

            // 取得できた骨格毎にループ
            for (int skelIdx = 0; skelIdx < skeletonBuffer.Length; ++skelIdx)
            {
                Skeleton skeleton = skeletonBuffer[skelIdx];

                // トラッキングできない骨格は処理しない
                if (skeleton.TrackingState != SkeletonTrackingState.Tracked)
                    continue;

                // ジェスチャーのチェック
                Tuple<int, int> res
                    = checkGesture(skeleton, gestureState[skelIdx],
                                   prevSkeleton[skelIdx]);
                gestureState[skelIdx] = res.Item2;
                prevSkeleton[skelIdx] = skeleton;

                // ポーズのチェック
                int pose = checkPosture(skeleton);

                // 骨格から頭を取得
                Joint head = skeleton.Joints[JointType.Head];

                // 頭の位置が取得できない状態の場合は処理しない
                if (head.TrackingState != JointTrackingState.Tracked
                    && head.TrackingState != JointTrackingState.Inferred)
                    continue;

                // 頭の位置を保存
                //results.Add(head.Position);

                // 頭の向きを取得
                Matrix4 headMtrx = (skeleton.BoneOrientations[JointType.Head]
                                    .AbsoluteRotation.Matrix);

                // 頭の位置・向きを保存
                results.Add(Tuple.Create(head.Position, headMtrx, res.Item1));
            }
            return results;
        }

        // RGBカメラの画像情報に、顔の位置にマスクを上書きして描画する
        private void fillBitmap(KinectSensor kinect, ColorImageFrame imgFrame,
                                DepthImageFrame depthFrame,
                                List<Tuple<SkeletonPoint, Matrix4, int>> headList)
        {
            // 描画の準備
            var drawContext = drawVisual.RenderOpen();
            int frmWidth = imgFrame.Width;
            int frmHeight = imgFrame.Height;
            int depWidth = depthFrame.Width;
            int depHeight = depthFrame.Height;

            // 画像情報をバッファにコピー
            imgFrame.CopyPixelDataTo(pixelBuffer);
            depthFrame.CopyPixelDataTo(depthBuffer);

            // 深度情報の各点に対する画像情報上の座標を取得
            kinect.MapDepthFrameToColorFrame(depthFormat, depthBuffer,
                                             rgbFormat, clrPntBuffer);

            // 深度情報の各点を調べ、プレイヤーがいない場合薄く青を塗る
            Array.Clear(depMaskBuffer, 0, depMaskBuffer.Length);
            for (int depIdx = 0; depIdx < depthBuffer.Length; ++depIdx)
            {
                if ((depthBuffer[depIdx] & DepthImageFrame.PlayerIndexBitmask) == 0)
                {
                    ColorImagePoint clrPoint = clrPntBuffer[depIdx];
                    int clrIdx = clrPoint.Y * frmWidth + clrPoint.X;
                    depMaskBuffer[clrIdx * 4] = 0;
                    depMaskBuffer[clrIdx * 4 + 1] = 0;
                    depMaskBuffer[clrIdx * 4 + 2] = 0;
                    depMaskBuffer[clrIdx * 4 + 3] = 0;
                }
            }
            var bgMask = new WriteableBitmap(frmWidth, frmHeight, 96, 96,
                                             PixelFormats.Bgra32, null);
            bgMask.WritePixels(new Int32Rect(0, 0, frmWidth, depHeight),
                               depMaskBuffer, frmWidth * 4, 0);

            // カメラの画像情報から背景のビットマップを作成し描画
            var bgImg = new WriteableBitmap(frmWidth, frmHeight, 96, 96,
                                            PixelFormats.Bgr32, null);
            bgImg.WritePixels(new Int32Rect(0, 0, frmWidth, frmHeight),
                              pixelBuffer, frmWidth * 4, 0);

            //drawContext.DrawImage(bgImg, new Rect(0, 0, frmWidth, frmHeight));
            Rect frmRect = new Rect(0, 0, frmWidth, frmHeight);
            drawContext.DrawImage(bgImg, frmRect);
            drawContext.DrawImage(bgMask, frmRect);

            // getHeadPointsで取得した各頭部（の位置）毎にループ
            for (int idx = 0; headList != null && idx < headList.Count; ++idx)
            {
                // 骨格の座標から画像情報の座標に変換
                SkeletonPoint headPos = headList[idx].Item1;
                ColorImagePoint headPt
//                    = kinect.MapSkeletonPointToColor(headList[idx], rgbFormat);
                      = kinect.MapSkeletonPointToColor(headPos, rgbFormat);

                // 距離に応じてサイズを決定
                int size = (int)(192 / headPos.Z);

                // ポーズをとると頭のサイズを２倍にする
                if (headList[idx].Item3 == 1)
                    size *= 3;

                // 頭の位置にマスク画像を描画
                //Rect rect = new Rect(headPt.X - 64, headPt.Y - 64, 128, 128);
                //drawContext.DrawImage(maskImage, rect);
                // 頭の位置に頭の向きに回転させたマスク画像を描画
                Matrix4 headMtrx = headList[idx].Item2;
                Matrix rot = new Matrix(headMtrx.M11, -headMtrx.M12,
                                        -headMtrx.M21, headMtrx.M22,
                                        headPt.X, headPt.Y);
                drawContext.PushTransform(new MatrixTransform(rot));
                //Rect rect = new Rect(-64, -64, 128, 128);
                Rect rect = new Rect(-size / 2, -size / 2, size, size);
                drawContext.DrawImage(maskImage, rect);
                drawContext.Pop();

                // プレイヤーの方向が音源の方向と近い場合は吹き出しを描画
                double angle = Math.Atan2(headPos.X, headPos.Z) * 180 / Math.PI;
                if (Math.Abs(soundDir - angle) < 10)
                {
                    Rect frect = new Rect(headPt.X + 32, headPt.Y - 64, 192, 128);
                    drawContext.DrawImage(fukidashiImage, frect);

                    // 音声を認識している場合はその文を吹き出しに表示
                    if (recognizedText != null)
                    {
                        var text = new FormattedText(recognizedText,
                                                     CultureInfo.GetCultureInfo("ja-JP"),
                                                     FlowDirection.LeftToRight,
                                                     new Typeface("Verdana"),
                                                     24, Brushes.Black);
                        var pt = new Point(headPt.X + 70, headPt.Y - 10);
                        drawContext.DrawText(text, pt);
                    }
                }
            }

            // 画像に表示するビットマップに描画
            drawContext.Close();
            bmpBuffer.Render(drawVisual);

            GC.Collect();
        }

        // 音声認識エンジンを初期化、文法を登録して返す
        private SpeechRecognitionEngine InitSpeechEngine()
        {
            RecognizerInfo targetRi = null;
            foreach (RecognizerInfo recognizer
                     in SpeechRecognitionEngine.InstalledRecognizers())
            {
                if (recognizer.AdditionalInfo.ContainsKey("Kinect")
                    && "True".Equals(recognizer.AdditionalInfo["Kinect"],
                                     StringComparison.OrdinalIgnoreCase)
                    && "ja-JP".Equals(recognizer.Culture.Name,
                                      StringComparison.OrdinalIgnoreCase))
                {
                    targetRi = recognizer;
                    break;
                }
            }
            if (targetRi == null)
                return null;
            SpeechRecognitionEngine engine
                = new SpeechRecognitionEngine(targetRi.Id);

            // 認識する単語の追加
            var words = new Choices();
            words.Add("キネクト");
            words.Add("テスト");
            words.Add("カズジービー");

            var grammarBuilder = new GrammarBuilder();
            grammarBuilder.Culture = targetRi.Culture;
            grammarBuilder.Append(words);
            var grammar = new Grammar(grammarBuilder);
            engine.LoadGrammar(grammar);

            return engine;
        }

        private void WindowClosed(object sender, EventArgs e)
        {
            kinectChooser.Stop();
        }

        //private void ColorImageReady(object sender, ColorImageFrameReadyEventArgs e)
        //{
        //    using (ColorImageFrame imageFrame = e.OpenColorImageFrame())
        //    {
        //        if (imageFrame != null)
        //        {
        //            // 画像情報の幅・高さ取得　※途中で変わらない想定！
        //            int frmWidth = imageFrame.Width;
        //            int frmHeight = imageFrame.Height;

        //            // 画像情報をバッファにコピー
        //            imageFrame.CopyPixelDataTo(pixelBuffer);
        //            // ビットマップに描画
        //            Int32Rect src = new Int32Rect(0, 0, frmWidth, frmHeight);
        //            bmpBuffer.WritePixels(src, pixelBuffer, frmWidth * 4, 0);

        //        }

        //    }
        //}

        // 音声認識エンジンの終了処理
        private void UninitSpeechEngine()
        {
            if (speechEngine != null)
            {
                speechEngine.SpeechRecognized -= SpeechRecognized;
                speechEngine.Dispose();
                speechEngine = null;
            }
        }
        // 引数で渡した骨格がとるポーズを判定
        // 1: 右手を右上に上げる
        // 2: 右手を右下に下げる
        // 3: 右手を左下に下げる
        // 4: 右手を左上に上げる
        // 0: それ以外
        private int checkPosture(Skeleton skeleton)
        {
            // 必要な関節・方向を取得
            Joint head = skeleton.Joints[JointType.Head];
            Joint rwrist = skeleton.Joints[JointType.WristRight];
            Joint center = skeleton.Joints[JointType.HipCenter];


            // 一つでも位置がとれない場合は処理しない
            if ((head.TrackingState != JointTrackingState.Tracked
                  && head.TrackingState != JointTrackingState.Inferred)
                || (rwrist.TrackingState != JointTrackingState.Tracked
                    && rwrist.TrackingState != JointTrackingState.Inferred)
                || (center.TrackingState != JointTrackingState.Tracked
                    && center.TrackingState != JointTrackingState.Inferred))
                return 0;

            // 頭・右手首・腰の位置からポーズを判定
            if ((head.Position.Y < rwrist.Position.Y)
                && (head.Position.X < rwrist.Position.X))
                return 1;
            else if ((rwrist.Position.Y < center.Position.Y)
                && (center.Position.X < rwrist.Position.X))
                return 2;
            else if ((rwrist.Position.Y < center.Position.Y)
                && (rwrist.Position.X < center.Position.X))
                return 3;
            else if ((head.Position.Y < rwrist.Position.Y)
                && (rwrist.Position.X < head.Position.X))
                return 4;
            return 0;
        }


        // 引数 prevSkeleton, skeletonで渡した骨格が、
        // 引数 state の遷移状態の条件を満たすことを確認する
        private bool checkTransState(Skeleton skeleton, int state,
                                    Skeleton prevSkeleton)
        {

            if (prevSkeleton == null)
                return false;

            // 必要な関節・方向を取得
            Joint head = skeleton.Joints[JointType.Head];
            Joint rwrist = skeleton.Joints[JointType.WristRight];
            Joint center = skeleton.Joints[JointType.HipCenter];
            Joint prevrw = prevSkeleton.Joints[JointType.WristRight];


            // 一つでも位置がとれない場合は処理しない
            if ((head.TrackingState != JointTrackingState.Tracked
                  && head.TrackingState != JointTrackingState.Inferred)
                || (rwrist.TrackingState != JointTrackingState.Tracked
                    && rwrist.TrackingState != JointTrackingState.Inferred)
                || (center.TrackingState != JointTrackingState.Tracked
                    && center.TrackingState != JointTrackingState.Inferred)
                || (prevrw.TrackingState != JointTrackingState.Tracked
                    && prevrw.TrackingState != JointTrackingState.Inferred))
                return false;

            switch (state)
            {
                case 1:
                    // 右上から右下に遷移
                    // ⇒ 「右手首」が右にあり、直前の姿勢より下にあること
                    if ((center.Position.X < rwrist.Position.X)
                        && (rwrist.Position.Y < prevrw.Position.Y + MOVE_PLAY))
                        return true;
                    break;
                case 2:
                    // 右下から左下に遷移
                    // ⇒ 「右手首」が下にあり、直前の姿勢より左にあること
                    if ((rwrist.Position.Y < center.Position.Y)
                        && (rwrist.Position.X < prevrw.Position.X + MOVE_PLAY))
                        return true;
                    break;
                case 3:
                    // 左下から左上に遷移
                    // ⇒ 「右手首」が左にあり、直前の姿勢より上にあること
                    if ((rwrist.Position.X < center.Position.X)
                        && (prevrw.Position.Y < rwrist.Position.Y + MOVE_PLAY))
                        return true;
                    break;
                case 4:
                    // 左上から右上に遷移
                    // ⇒ 「右手首」が上にあり、直前の姿勢より右にあること
                    if ((head.Position.Y < rwrist.Position.Y)
                        && (prevrw.Position.X < rwrist.Position.X + MOVE_PLAY))
                        return true;
                    break;
                default:
                    break;
            }
            // 遷移状態の条件を満たさない
            return false;
        }

        // 引数で渡す骨格が行ったジェスチャーを判定
        private Tuple<int, int> checkGesture(Skeleton skeleton, int state,
                                             Skeleton prevSkeleton)
        {
            int gesture = 0;

            // 遷移状態の条件を満たさない場合初期状態に戻す
            int nextState;
            if (checkTransState(skeleton, state, prevSkeleton))
                nextState = state;
            else
                nextState = 0;

            // 現在のポーズを取得
            int posture = checkPosture(skeleton);

            switch (state)
            {
                case 0:
                    if (posture == 1)
                        nextState = 1;
                    break;
                case 1:
                    if (posture == 2)
                        nextState = 2;
                    break;
                case 2:
                    if (posture == 3)
                        nextState = 3;
                    break;
                case 3:
                    if (posture == 4)
                        nextState = 4;
                    break;
                case 4:
                    if (posture == 1)
                        nextState = 5;
                    break;
                case 5:
                    if (posture == 1)
                    {
                        gesture = 1;
                        nextState = 5;
                    }
                    break;
                default:
                    break;
            }
            return Tuple.Create(gesture, nextState);
        }
    }
}
